function [r,a,Nd] = spherical_shell_distribution_dipoles(R1, R2, N, pos0)

% This function obtains the position of the dipoles resulting from
% discretizing a in a cubic net a spherical shell in N punctual dipoles.

%%%%%% INPUT ARGUMENTS %%%%%%
%%% R1 and R2: inner and outer radius or viceversa                  %%%
%%% N:         number of dipoles inside the scatterer               %%%
%%% pos0:      position of the center of mass of the parallelepiped %%%

%%%%%% OUTUT ARGUMENTS %%%%%%
%%% r: (Nd,3) vector containing the 3D positions of the Nd dipoles  %%%
%%% a:        lattice parametwr of the resulting net                %%%

%%% Organizing limits %%%
if(R1 > R2)
    
    aux = R2;
    R2  = R1;
    R1  = aux;
    
end

a        = ((4*pi*(R2^3-R1^3))/(3*N))^(1/3); %%% Lattice constant

%%%% Generation of coordinates (solid parallelepiped)%%%%
x = -R2:a:R2;
y = x;
z = x;

[XXX,YYY,ZZZ] = meshgrid(x,y,z);
X(:) = XXX(1:numel(XXX));
Y(:) = YYY(1:numel(YYY));
Z(:) = ZZZ(1:numel(ZZZ));

%%% Centering the distribution on the origin %%%
aux =  abs([min(X),max(X)]);
dX  =  max(aux-mean(aux));
if(max(X) < mean(aux))
    X = X+dX;
else
    X = X-dX;
end

aux =  abs([min(Z),max(Z)]);
dZ  =  max(aux-mean(aux));
if(max(Z) < mean(aux))
    Z = Z+dZ;
else
    Z = Z-dZ;
end

aux =  abs([min(Y),max(Y)]);
dY  =  max(aux-mean(aux));
if(max(Y) < mean(aux))
    Y = Y+dY;
else
    Y = Y-dY;
end

%%% Discarding inner and outer positions of the shell %%%
k = 0;
for i = 1:length(X)
    
    cond1 = X(i-k)^2 + Y(i-k)^2 + Z(i-k)^2 < R1^2;
    cond2 = X(i-k)^2 + Y(i-k)^2 + Z(i-k)^2 > R2^2;
    
    if( cond1 || cond2 )
        X(i-k)  = [];
        Y(i-k)  = [];
        Z(i-k)  = [];
        k       = k+1;
    end
    
end

%%%% Displacing the center of mass %%%%
r(:,1) = X(:)+pos0(1);
r(:,2) = Y(:)+pos0(2);
r(:,3) = Z(:)+pos0(3);

%%%% Resulting number of dipoles %%%%
Nd = length(r(:,1));

end